# traverse_dpt_tape_files.py
# Copyright 2023 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""Traverse TAPE files generated by DPT Unload() or compatible methods.

DPT Unload() and Load() methods are intended for DPT file reorganization.

The DPT DBA Guide describes the TAPE file formats and states the Load()
method will accept TAPE files in the correct format whatever their source.

This module verifies the structure by walking though the files.
"""

import tkinter
import tkinter.messagebox
import tkinter.filedialog
import os

STARS70 = b"*" * 70
DPT_BITMAP_BYTES = 8160
DPT_BITMAP_SIZE = DPT_BITMAP_BYTES * 8
DPT_BITMAP_FLAG = 65535  # int for 2 byte 'xff\xff' flag.


def _seek_data_start(filepath):
    """Return open files positioned at end of leading comment lines."""
    tape = open(filepath, "rb")  # Not with ... because open file returned.
    line = tape.read(70)
    if line == STARS70:
        for i in range(5):
            tape.readline()
        del i
    else:
        tape.seek(0)
    return tape


def _report_different(filepath1, filepath2):
    """Generate files differ report."""
    print(
        os.path.basename(filepath1),
        "and",
        os.path.basename(filepath2),
        "differ in content",
    )


def _traverse_tapef_file(filepath):
    """Do nothing."""
    del filepath


def _extract_records(data):
    r"""Verify TAPED file by following pointers to EOF.

    Format is 4 byte record number followed by optional 2 byte block
    transfer marker then field==value pairs until b'\xff\xff' record
    terminator.  Field name can be 2 byte field code or 1 byte field length
    and field name.  Field value in 8 bytes for FLOAT, 1 byte value length
    and value for STRING, and 4 byte value length and value for BLOB.

    Allow for STRING values only.

    """
    record_count = 0
    if len(data) == 0:
        return record_count
    position = 0
    while True:
        if position >= len(data):
            return record_count
        bytes_ = data[position : position + 4]
        if len(bytes_) != 4:
            print("record number expected at position", position)
            return record_count
        position += 4
        while True:
            bytes_ = data[position : position + 2]
            if len(bytes_) != 2:
                print(
                    "field code or block xfer expected at position",
                    position,
                )
                return record_count
            if bytes_ == b"\xff\xff":
                position += 2
                break

            # Allow for block transfer marker.
            if int.from_bytes(bytes_, "little", signed=True) < 0:
                position += 2
                bytes_ = data[position : position + 2]
                if len(bytes_) != 2:
                    print(
                        "field code or name expected at position",
                        position,
                    )
                    return record_count

            # Field codes are 0 to 4000, and first character of field name
            # is upper case alpha.
            if int.from_bytes(bytes_, "little") > 4000:
                position += 2
                length = bytes_[0]
                bytes_ = data[position : position + length - 1]
                if len(bytes_) != length - 1:
                    print("field value expected at position", position - 1)
                    return record_count
                position += len(bytes_)
            else:
                position += 2

            #  Handles STRING values only.
            bytes_ = data[position : position + 1]
            if len(bytes_) != 1:
                print("value length expected at position", position)
                return record_count
            position += 1
            length = bytes_[0]
            bytes_ = data[position : position + length]
            if len(bytes_) != length:
                print("unexpected value length at position", position)
                return record_count
            position += length
        record_count += 1
    return record_count


def _traverse_taped_file(filepath):
    """Skip leading comments and call _extract_records to verify data."""
    tape = _seek_data_start(filepath)
    data = tape.read()
    records = _extract_records(data)
    print(records, "in", os.path.basename(filepath), "\n")


def _extract_values(data, filepath):
    """Verify TAPEI file by following pointers to EOF."""
    values = {}
    if len(data) == 0:
        return values
    position = 0
    while True:
        if position >= len(data):
            return values
        while True:
            value_length = data[position]
            position += 1
            value = data[position : position + value_length]
            position += value_length
            if position >= len(data):
                print(
                    "position", position, "beyond end data after value length"
                )
                return values
            if value in values:
                print("duplicate", value, "at", position - value_length)
                return values
            values[value] = 0  # value in no segments initially
            bytes_ = data[position : position + 2]
            position += 2
            if len(bytes_) != 2:
                print("expecting 2 bytes value start at position", position)
                return values
            if bytes_ != b"\xff\xff":
                values[value] += 1
                # segnum = bytes_
                bytes_ = data[position : position + 4]
                if len(bytes_) != 4:
                    print(
                        "expecting 6 bytes at position ",
                        position - 2,
                        "for value with 1 record in segment",
                    )
                    return values
                recnum = int.from_bytes(bytes_[:2], "little")
                if recnum >= DPT_BITMAP_SIZE:
                    print(
                        "only record number",
                        recnum,
                        "within segment to high at position",
                        position,
                    )
                    return values
                position += 4
                break
            while True:
                bytes_ = data[position : position + 4]
                if len(bytes_) != 4:
                    print(
                        "".join(
                            (
                                "expecting 4 byte value terminator or ",
                                "record number list at position",
                            )
                        ),
                        position,
                    )
                    return values
                position += 4
                if bytes_ == b"\xff\xff\xff\xff":
                    break
                record_count = int.from_bytes(bytes_[-2:], "little")
                if record_count == DPT_BITMAP_FLAG:
                    values[value] += 1
                    position += DPT_BITMAP_BYTES
                elif record_count > DPT_BITMAP_SIZE:
                    print(
                        "expecting list count but",
                        record_count,
                        "at position",
                        position,
                        "is too big",
                    )
                    return values
                else:
                    values[value] += 1
                    record_count_bytes = record_count * 2
                    bytes_ = data[position : position + record_count_bytes]
                    if len(bytes_) != record_count_bytes:
                        print(
                            "expecting",
                            record_count_bytes,
                            "bytes of record numbers for",
                            record_count,
                            "records at position",
                            position,
                            "got",
                            len(bytes_),
                            "bytes",
                        )
                        return values
                    for offset in range(0, record_count_bytes, 2):
                        recnum = int.from_bytes(
                            bytes_[offset : offset + 2], "little"
                        )
                        if recnum >= DPT_BITMAP_SIZE:
                            print(
                                "record number",
                                recnum,
                                "within segment to high at position",
                                position,
                                "plus offset",
                                offset,
                                "in record number list",
                            )
                            return values
                    position += record_count_bytes
            if position >= len(data):
                if position > len(data):
                    print(
                        "position",
                        position,
                        "out of data",
                        len(data),
                        os.path.basename(filepath),
                    )
                break


def _traverse_tapei_file(filepath):
    """Skip leading comments and call _extract_values to verify indicies."""
    tape = _seek_data_start(filepath)
    data = tape.read()
    values = _extract_values(data, filepath)
    print(len(values), "values", os.path.basename(filepath), "\n")


def traverse_tape_directory(directory):
    """Verify all TAPED, TAPEF, and TAPEI, files in directory."""
    for file in os.listdir(path=directory):
        name, ext = os.path.splitext(file)
        del name
        if ext.upper() != ".DAT":
            continue
        if file.upper().endswith("TAPED.DAT"):
            _traverse_taped_file(os.path.join(directory, file))
            continue
        if file.upper().endswith("TAPEF.DAT"):
            _traverse_tapef_file(os.path.join(directory, file))
            continue
        if "_TAPEI_" in file.upper():
            _traverse_tapei_file(os.path.join(directory, file))
    print("TAPE files in directory", directory, "traversed")


if __name__ == "__main__":
    tape_directory = tkinter.filedialog.askdirectory(
        title="Open TAPE file directory",
        initialdir="~",
    )
    if tape_directory:
        traverse_tape_directory(tape_directory)
