# compare_dpt_tape_files.py
# Copyright 2023 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""Compare TAPE files generated by DPT Unload() or compatible methods.

DPT Unload() and Load() methods are intended for DPT file reorganization.

The DPT DBA Guide describes the TAPE file formats and states the Load()
method will accept TAPE files in the correct format whatever their source.

This module allows comparison of files generated by DPT Unload() and files
from other sources.  It likely has just enough capability to assist solving
the most recent problem encountered generating TAPE files by a non-DPT
method.
"""

import tkinter
import tkinter.messagebox
import tkinter.filedialog
import os

STARS70 = b"*" * 70


def _seek_data_start(filepath1, filepath2):
    """Return open files positioned at end of leading comment lines."""
    files = []
    for path in (filepath1, filepath2):
        tape = open(path, "rb")  # Not with ... because open file returned.
        line = tape.read(70)
        if line == STARS70:
            for i in range(5):
                tape.readline()
            del i
        else:
            tape.seek(0)
        files.append(tape)
    return files


def _report_different(filepath1, filepath2):
    """Generate files differ report."""
    print(
        os.path.basename(filepath1),
        "and",
        os.path.basename(filepath2),
        "differ in content",
    )


def _compare_tapef_files(filepath1, filepath2):
    """Compare TAPEF files in directories filepath1 and filepath2."""
    tape1, tape2 = _seek_data_start(filepath1, filepath2)
    data1 = tape1.read()
    data2 = tape2.read()
    if data1 != data2:
        _report_different(filepath1, filepath2)


def _extract_records(data):
    r"""Go.

    Format is 4 byte record number followed by optional 2 byte block
    transfer marker then field==value pairs until b'\xff\xff' record
    terminator.  Field name can be 2 byte field code or 1 byte field length
    and field name.  Field value in 8 bytes for FLOAT, 1 byte value length
    and value for STRING, and 4 byte value length and value for BLOB.

    Allow for STRING values only.

    """
    fields = []
    if len(data) == 0:
        return fields
    position = 0
    while True:
        if position >= len(data):
            return fields
        bytes_ = data[position : position + 4]
        if len(bytes_) != 4:
            print("record number expected at position", position)
            return None
        position += 4
        while True:
            bytes_ = data[position : position + 2]
            if len(bytes_) != 2:
                print(
                    "field code or block xfer expected at position",
                    position,
                )
                return None
            if bytes_ == b"\xff\xff":
                position += 2
                break

            # Allow for block transfer marker.
            if int.from_bytes(bytes_, "little", signed=True) < 0:
                position += 2
                bytes_ = data[position : position + 2]
                if len(bytes_) != 2:
                    print(
                        "field code or name expected at position",
                        position,
                    )
                    return None

            # Field codes are 0 to 4000, and first character of field name
            # is upper case alpha.
            if int.from_bytes(bytes_, "little") > 4000:
                position += 2
                length = bytes_[0]
                bytes_ = data[position : position + length - 1]
                if len(bytes_) != length - 1:
                    print("field value expected at position", position - 1)
                    return None
                position += len(bytes_)
            else:
                position += 2

            #  Handles STRING values only.
            bytes_ = data[position : position + 1]
            if len(bytes_) != 1:
                print("value length expected at position", position)
                return None
            position += 1
            length = bytes_[0]
            bytes_ = data[position : position + length]
            if len(bytes_) != length:
                print("unexpected value length at position", position)
                return None
            fields.append(bytes_)
            position += length
    return fields


def _compare_taped_files(filepath1, filepath2):
    """Compare TAPED files in directories filepath1 and filepath2."""
    tape1, tape2 = _seek_data_start(filepath1, filepath2)
    data1 = tape1.read()
    data2 = tape2.read()
    if data1 != data2:
        _report_different(filepath1, filepath2)
        records = []
        for data in (data1, data2):
            records.append(_extract_records(data))
        print("field counts", len(records[0]), len(records[1]))
        same = 0
        different = 0
        for record1, record2 in zip(records[0], records[1]):
            if record1 == record2:
                same += 1
            else:
                different += 1
        print(same, "same", "   ", different, "different")


def _extract_values(data):
    """Extract visible field occurrences from data."""
    values = {}
    if len(data) == 0:
        return values
    position = 0
    while True:
        if position >= len(data):
            return values
        while True:
            value_length = data[position]
            position += 1
            value = data[position : position + value_length]
            position += value_length
            if position >= len(data):
                return None
            if value in values:
                print("duplicate", value)
                return None
            bytes_ = data[position : position + 2]
            position += 2
            if len(bytes_) != 2:
                return None
            if bytes_ == b"\x00\x00":
                values[value] = 1
                bytes_ = data[position : position + 4]
                if len(bytes_) != 4:
                    return None
                position += 4
                break
            if bytes_ != b"\xff\xff":
                print("unknown segment count marker expected")
                return None
            while True:
                bytes_ = data[position : position + 4]
                if len(bytes_) != 4:
                    return None
                position += 4
                if bytes_ == b"\xff\xff\xff\xff":
                    break
                record_count = int.from_bytes(bytes_[-2:], "little")
                record_count_bytes = record_count * 2
                bytes_ = data[position : position + record_count_bytes]
                if len(bytes_) != record_count_bytes:
                    return None
                position += record_count_bytes


def _compare_tapei_files(filepath1, filepath2):
    """Compare TAPEI files in directories filepath1 and filepath2."""
    tape1, tape2 = _seek_data_start(filepath1, filepath2)
    data1 = tape1.read()
    data2 = tape2.read()
    if data1 != data2:
        _report_different(filepath1, filepath2)
        values = []
        for data in (data1, data2):
            values.append(_extract_values(data))
        print("value counts", len(values[0]), len(values[1]))
        set0 = set(values[0])
        set1 = set(values[1])
        intersection = set0.intersection(set1)
        print("value count in intersection", len(intersection))
        set0.difference_update(intersection)
        set1.difference_update(intersection)
        print(sorted(set0)[-3:])
        print(sorted(set1)[-3:])


def compare_tape_directories(directory1, directory2):
    """Compare the TAPE files in directory1 and directory2."""
    if directory1 == directory2:
        tkinter.messagebox.showinfo(
            title="Compare TAPE files",
            message=os.path.basename(directory1).join(
                ("No point comparing ", " with itself")
            ),
        )
        return
    filelist2 = set(os.listdir(path=directory2))
    for file1 in os.listdir(path=directory1):
        name, ext = os.path.splitext(file1)
        del name
        if ext != ".DAT":
            continue
        for file2 in filelist2:
            name, ext = os.path.splitext(file2)
            if ext != ".DAT":
                continue
            if file1.upper() != file2.upper():
                continue
            if file1.upper().endswith("TAPED.DAT"):
                _compare_taped_files(
                    os.path.join(directory1, file1),
                    os.path.join(directory2, file2),
                )
                continue
            if file1.upper().endswith("TAPEF.DAT"):
                _compare_tapef_files(
                    os.path.join(directory1, file1),
                    os.path.join(directory2, file2),
                )
                continue
            _compare_tapei_files(
                os.path.join(directory1, file1),
                os.path.join(directory2, file2),
            )
    print("TAPE files in directories compared")


if __name__ == "__main__":
    first_directory = tkinter.filedialog.askdirectory(
        title="Open First TAPE file directory",
        initialdir="~",
    )
    if first_directory:
        second_directory = tkinter.filedialog.askdirectory(
            title="Open Second TAPE file directory",
            initialdir=os.path.dirname(first_directory),
        )
        if second_directory:
            compare_tape_directories(first_directory, second_directory)
