# pgnify.py
# Copyright 2025 Roger Marsh
# Licence: See LICENCE (BSD licence)

"""Generate minimal input format PGN from tokens produced by lexer."""


class PGNify:
    """Output minimal input format PGN to open file."""

    newline = "\n"
    space = " "

    def __init__(self, file):
        """Note file for PGN produced from tokens generated by lexer."""
        self.file = file
        self._lexer = None

    def set_lexer(self, lexer):
        """Set the lexer generating tokens."""
        self._lexer = lexer

    def tagpair_pgn_tag(self, token):
        """Append PGN tag when PGN tag expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def tagpair_pawn_move(self, token):
        """Append pawn move when tag pair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_piece_move_full(self, token):
        """Append fully disambiguated piece move when tag pair expected."""
        self.file.write(token.replace("-", ""))
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_piece_move(self, token):
        """Append piece move when tag pair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_king_move(self, token):
        """Append king move when tag pair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_castles(self, token):
        """Append castles move when tag pair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_game_termination(self, token):
        """Append game termination when tagpair expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def tagpair_start_rav(self, token):
        """Append start RAV when tagpair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_end_rav(self, token):
        """Append end RAV when tagpair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_nag(self, token):
        """Append numeric annotation glyph when tagpair expected."""
        self.file.write(token)
        self.file.write(self.space)
        self._lexer.set_actions(self._lexer.movetext_actions)

    def tagpair_comment(self, token):
        """Append comment when tagpair expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def tagpair_eol_comment(self, token):
        """Append comment to end of line when tagpair expected."""
        self.file.write(token)

    def tagpair_reserved(self, token):
        """Append reserved when tagpair expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def tagpair_match_starter(self, token):
        """Append character which starts PGN match when tagpair expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def tagpair_anything_else(self, token):
        """Append characters otherwise unmatches when tagpair expected."""
        self.file.write(token)
        self.file.write(self.newline)

    def movetext_pgn_tag(self, token):
        """Append PGN tag when movetext expected."""
        self.file.write(self.newline)
        self.file.write(token)
        self.file.write(self.newline)
        self._lexer.set_actions(self._lexer.tagpair_actions)

    def movetext_pawn_move(self, token):
        """Append pawn move when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_piece_move_full(self, token):
        """Append fully disambiguated piece move when movetext expected."""
        self.file.write(token.replace("-", ""))
        self.file.write(self.space)

    def movetext_piece_move(self, token):
        """Append piece move when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_king_move(self, token):
        """Append king move when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_castles(self, token):
        """Append castles move when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_game_termination(self, token):
        """Append game termination when movetext expected."""
        self.file.write(token)
        self.file.write(self.newline)
        self._lexer.set_actions(self._lexer.tagpair_actions)

    def movetext_start_rav(self, token):
        """Append start RAV when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_end_rav(self, token):
        """Append end RAV when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_nag(self, token):
        """Append numeric annotation glyph when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_comment(self, token):
        """Append comment when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_eol_comment(self, token):
        """Append comment to end of line when movetext expected."""
        self.file.write(token)

    def movetext_reserved(self, token):
        """Append reserved when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_match_starter(self, token):
        """Append character which starts PGN match when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)

    def movetext_anything_else(self, token):
        """Append characters otherwise unmatches when movetext expected."""
        self.file.write(token)
        self.file.write(self.space)
